;
;                       PureFrog
;                       --------
;
;    The FroggyHome sensor monitor written in PureBasic.
;
;         Written by:
;
;             - Frederic 'AlphaSND' Laboureur
;             - Chaumontet 'snip' Sebastien
;
;         Based on the 'frogd' linux client by Regis 'Redge' Barbier
;         http://www.redge.net/frogd/fr/
;
;
; Changes :
; 1.3 - 01/13/2003 (AlphaSND)
;      * Make it compile with PureBasic 3.51
;
; 1.2 - 02/24/2002 (snip & AlphaSND)
;      * Proxy feature added for WeatherByU
;      * User-Agent info
;      * Change About DialogBox
;
  
#FROG_RESET_DELAY = 2000
#SENSOR_PRES_TEMP = $10
#SENSOR_HR        = $30
#FROG_ALTITUDE    = 0

#FS_Initialization = 0
#FS_Measurement = 1


Structure DCB2
  DCBlength.l   ; sizeof(DCB) 
  BaudRate.l    ; current baud rate 
  Flags.l
  wReserved.w   ; not currently used 
  XonLim.w      ; transmit XON threshold 
  XoffLim.w     ; transmit XOFF threshold 
  ByteSize.b    ; number of bits/byte, 4-8 
  Parity.b      ; 0-4=no,odd,even,mark,space 
  StopBits.b    ; 0,1,2 = 1, 1.5, 2 
  XonChar.b     ; Tx and Rx XON character 
  XoffChar.b    ; Tx and Rx XOFF character 
  ErrorChar.b   ; error replacement character 
  EofChar.b     ; end of input character 
  EvtChar.b     ; received event character 
  wReserved1.w  ; reserved; do not use 
EndStructure


Structure Sensor
  Type.l
  PhyMesurementSize.l
  DataSize.l
  DataContent.s
EndStructure

Structure Frog
  Firmware.l
  PartNumber.l
  SerialNumber.l
  NbSensors.l
  Sensor.Sensor[10]
  LastTemperature.f
  LastPressure.f
  LastHumidity.f
  Altitude.f
EndStructure

Global Froggy.Frog, MeasurementDelay, FrogState, *File
Global EnableLog, EnableSystray, ShowGUI


Procedure.s AddZero(Number)
  Result$ = Str(Number)

  If Number < 10
    Result$ = "0"+Result$ 
  EndIf
  
  ProcedureReturn Result$
EndProcedure


Procedure SendWeatherData(ID$, Password$, Host$, Port, ProxyHost$, ProxyPort, UseProxy)
  Shared IsNetworkAvailable

  If IsNetworkAvailable <= 0 
    If InitNetwork() = 0
      If ShowGUI : MessageRequester("Error", "Network can't be initialized.", #MB_ICONERROR) : EndIf
      IsNetworkAvailable = -1
    Else
      IsNetworkAvailable = 1
    EndIf
  EndIf

  If IsNetworkAvailable > 0

    If UseProxy
      SockHost$ = ProxyHost$
      SockPort  = ProxyPort
    Else
      SockHost$ = Host$
      SockPort  = Port
    EndIf

    ConnectionID = OpenNetworkConnection(SockHost$, SockPort)
    If ConnectionID

      GetSystemTime_ (time.SYSTEMTIME)
      Time$ = Str(time\wYear)+"-"+AddZero(time\wMonth)+"-"+AddZero(time\wDay)+"+"+AddZero(time\wHour)+":"+AddZero(time\wMinute)+":"+AddZero(time\wSecond)
      
      Temperature.f = 32+((Froggy\LastTemperature)*9)/5
      Humidity.f    = Froggy\LastHumidity
      Pressure.f    = Froggy\LastPressure*0.02953
      
      EOL$ = Chr(13)+Chr(10)

      a$ = "GET http://"+Host$+":"+Str(Port)+"/WeatherUnderground?ID="+ID$+"&PASSWORD="+Password$+"&dateutc="+Time$+"&tempf="+StrF(Temperature,2)+"&baromin="+StrF(Pressure,6)+"&humidity="+StrF(Humidity,2)+" HTTP/1.1" + EOL$
      a$ + "Host: "+Host$+":"+Str(Port) + EOL$
      a$ + "User-Agent: PureFrog 1.2" + EOL$
      a$ + EOL$
      SendNetworkData(ConnectionID, a$, Len(a$))

      CloseNetworkConnection(ConnectionID)
    Else
      If ShowGUI : MessageRequester("Error", "Server "+SockHost$+":"+Str(SockPort)+" not found.", #MB_ICONERROR) : EndIf
    EndIf
    
  EndIf
EndProcedure


Procedure.l HexToDec(String.s)
  If Len(String) > 2:ProcedureReturn 0:EndIf
 
  If Asc(String) > 60
    Result.l = Asc(String) - 55 
  Else
    Result = Asc(String) - 48
  EndIf
 
  Result << 4
 
  String = Right(String,1)
  
  If Asc(String) > 60
    Result + Asc(String) - 55
  Else
    Result + Asc(String) - 48 
  EndIf
 
  ProcedureReturn Result
EndProcedure


; Decode the input string : translate from the 2 char format To the 1 byte
;
Procedure.l Decode(Input$, Size)

  *Memory = AllocateMemory(0, Size/2+1, 0)

  For k=1 To Size Step 2
    PokeB(*Memory, HexToDec(Mid(Input$, k, 2)))
    *Memory+1
  Next
  
  ProcedureReturn MemoryID()
EndProcedure


; ---------------------------------------------------------------
; Calculate the checksum of a frame
; ---------------------------------------------------------------

Procedure CalcCheckSum(Input$, FrameSize)
    
  a$ = Right(Input$, FrameSize-1)
    
  *Result = Decode(a$, (FrameSize-1))
  ResultLength = (FrameSize-1)/2
    
  Check = 'G'
   
  For k=1 To ResultLength
    b = PeekB(*Result) : *Result+1
    MOV eax, Check
    XOr eax, b
    MOV Check, eax
  Next
   
  ProcedureReturn Check
EndProcedure


; ---------------------------------------------------------------
; Send a frame To the frog And wait For the response.
; ---------------------------------------------------------------

Procedure.s AskFrog(Command$)

   Frame$ = "G"+Command$
   Check  = CalcCheckSum(Frame$, Len(Frame$))
   
   Check$ = Hex(Check)
   If Len(Check$) = 1
     Check$ = "0"+Check$
   EndIf
   
   Frame$ = Frame$ + Check$ + "Z"
   
   UseFile(0)
   WriteString(Frame$)
   
   Delay(5000)
   
   a$ = " "
   String$ = ""
   
   UseFile(0)
   While ReadData(@a$, 1) And a$ <> "Z"
     String$ + a$ 
     UseFile(0)
   Wend
   
   ProcedureReturn String$
EndProcedure


Procedure HandleError(Result, Text$)

  If Result = 0
    MessageRequester("Erreur", Text$, #MB_ICONERROR)  
    End
  EndIf
  
EndProcedure


Procedure CalculateTemperaturePressure(Result$)

  Frame$ = Mid(Result$, 6, 8)
  
  a$ = Froggy\Sensor[0]\DataContent
  
  Offset = 3
  Word1 = HexToDec(Mid(a$, Offset, 2)) << 8 : Offset+2
  Word1 | HexToDec(Mid(a$, Offset, 2)) : Offset+2
  
  Word2 = HexToDec(Mid(a$, Offset, 2)) << 8 : Offset+2
  Word2 | HexToDec(Mid(a$, Offset, 2)) : Offset+2 
  
  Word3 = HexToDec(Mid(a$, Offset, 2)) << 8 : Offset+2
  Word3 | HexToDec(Mid(a$, Offset, 2)) : Offset+2 

  Word4 = HexToDec(Mid(a$, Offset, 2)) << 8 : Offset+2
  Word4 | HexToDec(Mid(a$, Offset, 2)) : Offset+2
  
  C1 = Word1 & $FFFE
  C1 >> 1
   
  C2 = Word3 & $003F
  C2 << 6
  
  w = Word4 & $003F
  C2 | w
  
  C3 = Word4 & $FFC0
  C3 >> 6
   
  C4 = Word3 & $FFC0
  C4 >> 6

  C5 = Word1 & $0001
  C5 << 10
  w = Word2 & $FFC0
  w >> 6
  C5 | w

  C6 = Word2 & $003F
  
  Offset = 1
  D1 = HexToDec(Mid(Frame$, Offset, 2)) : Offset+2 
  D1 << 8;
  D1 | HexToDec(Mid(Frame$, Offset, 2)) : Offset+2

  D2 = HexToDec(Mid(Frame$, Offset, 2)) : Offset+2 
  D2 << 8;
  D2 | HexToDec(Mid(Frame$, Offset, 2)) : Offset+2 
   
  fUTI.f = 8*C5+20224
  If fUTI < D2 
    fDT.f   = D2-fUTI
    fTemp.f = (200+fDT*(C6+50)/1024)/10
  Else
    fDT=(D2-fUTI)-( ((D2-fUTI)/128)*((D2-fUTI)/128) )/4
    fTemp=(200+fDT*(C6+50)/1024+fDT/256)/10;
  EndIf

  fOFF.f=C2*4+((C4-512)*fDT)/4096
  fSENS.f=C1+(C3*fDT)/1024+24576
  fX.f=(fSENS*(D1-7168))/16384-fOFF
  fPress.f=fX*10/32+2500
  fPress.f=fPress/10
  fPress.f=fPress*Pow(10,Froggy\Altitude/19434)

  Froggy\LastTemperature = fTemp
  Froggy\LastPressure = fPress
EndProcedure


; ----------------------------------------------------------------------------------

Procedure CalculateHygrometry(Result$)

   fTemp.f =  Froggy\LastTemperature
   
   Frame$ = Mid(Result$, 6+8, 9)
   
   a$ = Froggy\Sensor[1]\DataContent
   
   Offset = 3
    
   NbBits = HexToDec(Mid(a$, Offset, 2)) : Offset+2

   Rs = HexToDec(Mid(a$, Offset, 2)) : Offset+2
   Rs << 8;
   Rs | HexToDec(Mid(a$, Offset, 2)) : Offset+2
   Rs << 8;
   Rs | HexToDec(Mid(a$, Offset, 2)) : Offset+2
   
   Rp=HexToDec(Mid(a$, Offset, 2)) : Offset+2
   Rp << 8;
   Rp | HexToDec(Mid(a$, Offset, 2)) : Offset+2
   Rp << 8;
   Rp | HexToDec(Mid(a$, Offset, 2)) : Offset+2
   
   Offset = 3
   Mesure = HexToDec(Mid(Frame$, Offset, 2)) : Offset+2;
   Mesure << 8;
   Mesure | HexToDec(Mid(Frame$, Offset, 2)) : Offset+2
   
   If Mesure = 0 Or Rp = 0
     fHum.f = 100
   Else : If Mesure >= 507
     fHum = 0
   Else

   r.f = Pow(2, NbBits)
   fImped.f=(Rs/1000)/((r/Mesure)-1-(Rs/Rp));

   fa.f = 1.0
   fb.f = Log10(fImped)
   fc.f = Pow(fb,2)
   fd.f = Pow(fb,3)
      
   fT2.f = Pow(fTemp,2)
   fT3.f = Pow(fTemp,3)
     
   DefType.f
    
   fA1 =  115.4564      : fA2 = -25.557588     : fA3 =  0.4595314     : fA4 =  0.2008904
   fB1 = -0.3066346     : fB2 =  -0.8884106    : fB3 =  0.3460639     : fB4 = -0.03408861
   fC1 = -0.02558339    : fC2 =   0.03352567   : fC3 = -0.01106885    : fC4 =  0.00090457
   fD1 =  0.00033866
   fD2 = -0.00036163
   fD3 =  0.00009868
   fD4 = -0.00000245
    

   DefType.l

   fHum = fa*fA1+fb*fA2+fc*fA3+fd*fA4
   fHum + fTemp*fa*fB1+fTemp*fb*fB2+fTemp*fc*fB3+fTemp*fd*fB4
   fHum + fT2*fa*fC1+fT2*fb*fC2+fT2*fc*fC3+fT2*fd*fC4
   fHum + fT3*fa*fD1+fT3*fb*fD2+fT3*fc*fD3+fT3*fd*fD4

   
   If fHum > 100 
     fHum = 100.0;
   EndIf
   
   If fHum < 0.0
     fHum = 0.0;
   EndIf
   
  EndIf
  EndIf
 
  Froggy\LastHumidity = fHum;
    
EndProcedure


Procedure AddLogEntry()
  Shared Froggy

  If EnableLog
    GetLocalTime_ (time.SYSTEMTIME)
    
    If UseFile(1)
    
      WriteString (Str(time\wDay)  + "/" + Str(time\wMonth) + "/"+Str(time\wYear) + ":")
      WriteString (Str(time\wHour) + "/" + Str(time\wMinute)+":")
      WriteStringN(StrF(Froggy\LastTemperature, 1) + ":" + StrF(Froggy\LastPressure, 1)+":"+StrF(Froggy\LastHumidity, 1))
      
      UseFile(0)  
    EndIf
  EndIf  
    
EndProcedure


Procedure RefreshMonitor()
  StatusBarText(0, 0, StrF(Froggy\LastTemperature,1)+" C"  , 4)
  StatusBarText(0, 1, StrF(Froggy\LastPressure,1)   +" hPa", 4)
  StatusBarText(0, 2, StrF(Froggy\LastHumidity,1)   +"% HR", 4)

  If EnableSystray  
    SysTrayIconToolTip(0, StrF(Froggy\LastTemperature,1)+" C"+", "+StrF(Froggy\LastPressure,1)+" hPa, "+StrF(Froggy\LastHumidity,1)+"% HR")
  EndIf

EndProcedure


Procedure FrogThread(Parameter)
  Shared EnableWBU, WBUStationID$, WBUPassword$, WBUServer$, WBUServerPort, FrogAltitude
  Shared ProxyServer$, ProxyPort, UseProxy

  Repeat
    
    If FrogState = #FS_Initialization
    
      ; Hard Reset
      ;
      EscapeCommFunction_(*File,#SETDTR)
      EscapeCommFunction_(*File,#CLRRTS)
     
      Delay(#FROG_RESET_DELAY)
     
      EscapeCommFunction_(*File,#CLRDTR)
      EscapeCommFunction_(*File,#SETRTS)
     
      Delay(#FROG_RESET_DELAY);
     
      HandleError( PurgeComm_(*File, #PURGE_TXCLEAR | #PURGE_RXCLEAR), "PurgeComm()")
   
      Result$ = AskFrog("00")
      
      If Result$
        Offset = 4
        Froggy\Firmware     = HexToDec(Mid(Result$, Offset, 2)) : Offset+2 
        Froggy\PartNumber   = HexToDec(Mid(Result$, Offset, 2)) : Offset+2
        Offset+8
        Froggy\NbSensors    = HexToDec(Mid(Result$, Offset, 2)) : Offset+2

        For k=0 To Froggy\NbSensors-1
          Froggy\Sensor[k]\Type              = HexToDec(Mid(Result$, Offset, 2)) : Offset+2
          Froggy\Sensor[k]\PhyMesurementSize = HexToDec(Mid(Result$, Offset, 2)) : Offset+2
          Froggy\Sensor[k]\DataSize          = HexToDec(Mid(Result$, Offset, 2)) : Offset+2
          a = Froggy\Sensor[k]\DataSize*2
          Froggy\Sensor[k]\DataContent       = Mid(Result$, Offset, a) : Offset+a
        Next

        FrogState = #FS_Measurement
      EndIf
      
    Else
      Result$ = AskFrog("01")
      If Result$
        Froggy\Altitude = FrogAltitude
        CalculateTemperaturePressure(Result$)
        CalculateHygrometry(Result$)
    
        AddLogEntry()
        
        If EnableWBU
          SendWeatherData(WBUStationID$, WBUPassword$, WBUServer$, WBUServerPort, ProxyServer$, ProxyPort, UseProxy)
        EndIf
        RefreshMonitor()
      Else
        FrogState = #FS_Initialization
      EndIf
    EndIf
    
    Delay((MeasurementDelay*60)*1000)
  
  ForEver

EndProcedure

EnableLog     = 1
EnableSystray = 1
ShowGUI       = 1

Repeat
  Parameter$ = UCase(ProgramParameter())

  Select Parameter$
    Case "/NOGUI"
      ShowGUI = 0
      
    Case "/NOSYSTRAY"
      EnableSystray = 0
      
    Case "/NOLOG"
      EnableLog = 0
      
  EndSelect

Until Paramater$ = ""

#GADGET_SelectPort    = 1
#GADGET_AddCustomPort = 2
#GADGET_Altitude      = 3

#GADGET_Sauvegarder   = 4
#GADGET_Annuler       = 5
#GADGET_Quitter       = 6

#GADGET_OffsetPressure    = 7
#GADGET_OffsetTemperature = 8
#GADGET_OffsetHumidity    = 9
#GADGET_Delay             = 10

#GADGET_MonitorT = 11
#GADGET_MonitorH = 12
#GADGET_MonitorP = 13

#GADGET_EnableWBU = 14
#GADGET_WBUOption = 15

#GADGET_WBUStationID  = 16
#GADGET_WBUPassword   = 17
#GADGET_WBUServer     = 18
#GADGET_WBUServerPort = 19

#GADGET_UseProxy      = 20
#GADGET_ProxyServer   = 21
#GADGET_ProxyPort     = 22

#GADGET_WBUOk         = 23
#GADGET_WBUCancel     = 24


Gosub LoadPrefs

If OpenWindow(1, MonitorX, MonitorY, 210, 40, #PB_Window_SystemMenu | #PB_Window_Invisible, "PureFrog 1.3 - Monitor")

  If CreateMenu(0, WindowID())
    MenuTitle("Project")
      MenuItem(1, "Configuration...")
      MenuBar()
      MenuItem(2, "Quit")
      
    MenuTitle("Help")
      MenuItem(3, "About...")
  EndIf
  
  
  If CreatePopupMenu(1)
    MenuItem(4, "Show Monitor")
    MenuItem(1, "Configuration")
    MenuBar()
    MenuItem(3, "About")
    MenuBar()
    MenuItem(2, "Quit")
  EndIf


  If CreateStatusBar(0, WindowID())
    AddStatusBarField(70)
    AddStatusBarField(70)
    AddStatusBarField(70)
    StatusBarText(0, 0, "...", 4)
    StatusBarText(0, 1, "...", 4)
    StatusBarText(0, 2, "...", 4)
  EndIf
      
  
  If EnableSystray
    AddSysTrayIcon(0, WindowID(), ExtractIcon_(GetModuleHandle_(0), "PureFrog.exe", 0))
    SysTrayIconToolTip(0, "Temperature: ... (-) Pressure: ... (-) Hygrometry: ...") 
  EndIf
  
  
  If ShowGUI
    ShowWindow_(WindowID(), #SW_SHOW)
  EndIf
  
  ShowGUI = 1 ; It's just for the start we don't need a gui...
  
  Gosub StartFrog
      
  Repeat
    Event = WaitWindowEvent()
    
    If EventWindowID() = 1      ; -------------------------------------- Monitor
      UseWindow(1)
      
      Select Event
        Case #PB_EventCloseWindow
          ShowWindow_(WindowID(), #SW_HIDE)
          ShowGUI = 0
        
        Case #PB_EventSysTray
          If EventType() = #PB_EventType_LeftClick
            ShowWindow_(WindowID(), #SW_SHOW)
            SetWindowPos_(WindowID(), #HWND_TOP, 0,0,0,0, #SWP_NOMOVE | #SWP_NOSIZE)
            ShowGUI = 1
          Else
            GetCursorPos_(Cursor.POINT)
            SetForegroundWindow_(WindowID())
            DisplayPopupMenu(1, WindowID(), Cursor\x, Cursor\y)
          EndIf
          
        Case #PB_EventMenu
          Select EventMenuID()
          
            Case 1
              Gosub OpenConfigurationWindow
            
            Case 2
              Quit = 1
          
            Case 3
              MessageRequester("Information", "PureFrog 1.2 - Coded in PureBasic"+Chr(10)+"By Fred 'AlphaSND' Laboureur"+Chr(10)+"and Sebastien Chaumontet 'Snip'"+Chr(10)+Chr(10)+"Firmware: "+Str(Froggy\Firmware)+Chr(10)+"PartNumber: "+Str(Froggy\PartNumber)+Chr(10)+Chr(10)+"Sensors: "+Str(Froggy\NbSensors)+Chr(10)+Chr(10)+"Sensors by http://FroggyHome.com/", #MB_ICONINFORMATION)
            
            Case 4
              ShowWindow_(WindowID(), #SW_SHOW)
              SetWindowPos_(WindowID(), #HWND_TOP, 0,0,0,0, #SWP_NOMOVE | #SWP_NOSIZE)              
          EndSelect
          
      EndSelect
      
    Else:If EventWindowID() = 0 ; -------------------------------------- Configuration
      UseWindow(0)
    
      Select Event
        Case #PB_EventCloseWindow
          CloseWindow(0)
          
        Case #PB_EventGadget
          Select EventGadgetID()
          
            Case #GADGET_EnableWBU
              DisableGadget (#GADGET_WBUOption, 1-GetGadgetState(#GADGET_EnableWBU))
                            
            Case #GADGET_WBUOption
              Gosub OpenWBUOption
          
            Case #GADGET_Annuler
              CloseWindow(0)
            
            Case #GADGET_Sauvegarder
              Gosub SavePrefs
              CloseWindow(0)
              
          EndSelect
      EndSelect
    
    Else:If EventWindowID() = 2 ; -------------------------------------- WBU Option
      UseWindow(2)
    
      Select Event
        Case #PB_EventCloseWindow
          CloseWindow(2)
          
        Case #PB_EventGadget
          Select EventGadgetID()

            Case #GADGET_UseProxy
              DisableGadget (#GADGET_ProxyServer, 1-GetGadgetState(#GADGET_UseProxy))
              DisableGadget (#GADGET_ProxyPort, 1-GetGadgetState(#GADGET_UseProxy))

            Case #GADGET_WBUOk
              WBUStationID$ = GetGadgetText(#GADGET_WBUStationID)
              WBUPassword$  = GetGadgetText(#GADGET_WBUPassword)
              WBUServer$    = GetGadgetText(#GADGET_WBUServer)
              WBUServerPort = Val(GetGadgetText(#GADGET_WBUServerPort))
              ProxyServer$  = GetGadgetText(#GADGET_ProxyServer)
              ProxyPort     = Val(GetGadgetText(#GADGET_ProxyPort))
              UseProxy      = GetGadgetState(#GADGET_UseProxy)
              CloseWindow(2)
              
            Case #GADGET_WBUCancel
              CloseWindow(2)
                                                    
          EndSelect
      EndSelect
    
    EndIf
    EndIf
    EndIf
     
  Until Quit = 1
EndIf

End   
        

StartFrog:

  *File = OpenComPort(0, SelectedPort$+":")
  If *File

    If EnableLog
      If OpenFile(1, "PureFrog.log")
        FileSeek(Lof())
      EndIf
    EndIf
    
    If GetCommState_(*File, @PortConfig.DCB2)

      ; Allocation des tampons d'E/S
      ;
      HandleError( SetupComm_(*File, 4096, 4096), "SetupComm()" )
      
      ct.COMMTIMEOUTS
      ct\ReadIntervalTimeout         = #MAXDWORD
      ct\ReadTotalTimeoutMultiplier  = 0
      ct\ReadTotalTimeoutConstant    = 0
      ct\WriteTotalTimeoutMultiplier = 0
      ct\WriteTotalTimeoutConstant   = 0
      
      HandleError( SetCommTimeouts_(*File, ct), "SetCommTimeouts()" ) 
      
      ; construction des donnees d'initialisation du port
        
      dcb.DCB2;
    
      HandleError( GetCommState_(*File, @dcb), "GetCommState()" )
      
      dcb\BaudRate  = #CBR_300;
      dcb\Parity    = #NOPARITY;
      dcb\StopBits  = #TWOSTOPBITS;
      dcb\ByteSize  = 8;    
      dcb\Flags     = 4227  ; Combined flags values got from the C.. PureBasic doesn't support flags in structures
      
      HandleError( SetCommState_(*File, @dcb), "SetCommState()" )
      
      CreateThread(@FrogThread(), 0)
       
    EndIf
  Else
    MessageRequester("PureFrog","Can't open the following port: "+SelectedPort$+Chr(10)+"This port is may be in use", #MB_ICONERROR)
  EndIf
Return 
   

;-OpenConfiguration  

OpenConfigurationWindow:
    
  If OpenWindow(0, ConfigurationX, ConfigurationY, 268, 240, #PB_Window_SystemMenu, "Configuration")
  
    If CreateGadgetList(WindowID())
      Left = 10
      Top  = 10
      Height = 20
      TextGadget    (0, Left, Top+3, 100, Height, "ComPort :") : Left+60
      ComboBoxGadget(#GADGET_SelectPort   , Left, Top, 100, Height*20) : Left+110
      ButtonGadget  (#GADGET_AddCustomPort, Left, Top, 60, Height+2, "Add Port") : Top+Height+6 : Left = 10
      DisableGadget (#GADGET_AddCustomPort, 1)
      
      For k=1 To 8
        AddGadgetItem(#GADGET_SelectPort, -1, "COM"+Str(k))
      Next
      
      SetGadgetText(#GADGET_SelectPort, SelectedPort$)
      
      TextGadget  (0, Left, Top+3, 100, Height, "Altitude :") : Left+60
      StringGadget(#GADGET_Altitude, Left, Top, 100, Height, Str(FrogAltitude)) : Top+Height+6 : Left = 10
      
      TextGadget  (0, Left, Top+3, 100, Height, "Offset Pressure :") : Left+130
      StringGadget(#GADGET_OffsetPressure, Left, Top, 100, Height, Str(OffsetPressure)) : Top+Height+6 : Left = 10
      
      TextGadget  (0, Left, Top+3, 130, Height, "Offset Temperature :") : Left+130
      StringGadget(#GADGET_OffsetTemperature, Left, Top, 100, Height, Str(OffsetTemperature)) : Top+Height+6 : Left = 10

      TextGadget  (0, Left, Top+3, 100, Height, "Offset Humidity :") : Left+130
      StringGadget(#GADGET_OffsetHumidity, Left, Top, 100, Height, Str(OffsetHumidity)) : Top+Height+6 : Left = 10

      TextGadget  (0, Left, Top+3, 130, Height, "Delay in Minutes :") : Left+130
      StringGadget(#GADGET_Delay, Left, Top, 100, Height, Str(MeasurementDelay)) : Top+Height+6 : Left = 10
      
      CheckBoxGadget(#GADGET_EnableWBU, Left, Top, 120, Height, "Enable WeatherByU") : Left+130
      ButtonGadget  (#GADGET_WBUOption, Left, Top, 100, Height+2, "Option") : Top+Height+16 : Left = 10
            
      ButtonGadget(#GADGET_Sauvegarder, Left, Top, 120, Height+6, "Save")  : Left+125
      ButtonGadget(#GADGET_Annuler    , Left, Top, 120, Height+6, "Cancel")
    EndIf
    
    SetGadgetState(#GADGET_EnableWBU, EnableWBU)
    DisableGadget (#GADGET_WBUOption, 1-GetGadgetState(#GADGET_EnableWBU))
    
  EndIf
  
   
Return


;-OpenWBUOption

OpenWBUOption:
    
  If OpenWindow(2, ConfigurationX, ConfigurationY, 264, 260, #PB_Window_SystemMenu, "WeatherByU Option")
  
    If CreateGadgetList(WindowID())
      Left = 10
      Top  = 10
      Height = 20
      
      TextGadget  (0, Left, Top+3, 100, Height, "Station ID :") : Left+130
      StringGadget(#GADGET_WBUStationID, Left, Top, 100, Height, WBUStationID$) : Top+Height+6 : Left = 10
      
      TextGadget  (0, Left, Top+3, 100, Height, "Password :") : Left+130
      StringGadget(#GADGET_WBUPassword, Left, Top, 100, Height, WBUPassword$) : Top+Height+6 : Left = 10
      
      TextGadget  (0, Left, Top+3, 130, Height, "Server :") : Left+130
      StringGadget(#GADGET_WBUServer, Left, Top, 100, Height, WBUServer$) : Top+Height+6 : Left = 10

      TextGadget  (0, Left, Top+3, 100, Height, "Server Port :") : Left+130
      StringGadget(#GADGET_WBUServerPort, Left, Top, 100, Height, Str(WBUServerPort)) : Top+Height+6 : Left = 10

      CheckBoxGadget(#GADGET_UseProxy, Left, Top, 120, Height, "Use HTTP proxy") : Top+Height+6 : Left = 10

      TextGadget  (0, Left, Top+3, 130, Height, "Proxy Host :") : Left+130
      StringGadget(#GADGET_ProxyServer, Left, Top, 100, Height, ProxyServer$) : Top+Height+6 : Left = 10

      TextGadget  (0, Left, Top+3, 100, Height, "Proxy Port :") : Left+130
      StringGadget(#GADGET_ProxyPort, Left, Top, 100, Height, Str(ProxyPort)) : Top+Height+6 : Left = 10

      TextGadget  (0, Left+20, Top+3, 274, Height, "More infos : http://Weather.WeatherByU.com/") : Left=10 : Top+Height+16
      
      ButtonGadget(#GADGET_WBUOk    , Left, Top, 120, Height+6, "Ok")  : Left+125
      ButtonGadget(#GADGET_WBUCancel, Left, Top, 120, Height+6, "Cancel")
    EndIf

    SetGadgetState(#GADGET_UseProxy, UseProxy)
    DisableGadget (#GADGET_ProxyServer, 1-GetGadgetState(#GADGET_UseProxy))
    DisableGadget (#GADGET_ProxyPort, 1-GetGadgetState(#GADGET_UseProxy))

      
  EndIf
  
Return

;-SavePrefs

SavePrefs:

  SelectedPort$     = GetGadgetText(#GADGET_SelectPort)
  FrogAltitude      = Val(GetGadgetText(#GADGET_Altitude))
  OffsetPressure    = Val(GetGadgetText(#GADGET_OffsetPressure))
  OffsetTemperature = Val(GetGadgetText(#GADGET_OffsetTemperature))
  OffsetHumidity    = Val(GetGadgetText(#GADGET_OffsetHumidity))
  MeasurementDelay  = Val(GetGadgetText(#GADGET_Delay))
  EnableWBU         = GetGadgetState(#GADGET_EnableWBU)

  If CreateFile(2, "PureFrog.prefs")
   
    WriteStringN(SelectedPort$)
    WriteStringN(Str(FrogAltitude))
    WriteStringN(Str(OffsetPressure)) 
    WriteStringN(Str(OffsetTemperature))
    WriteStringN(Str(OffsetHumidity))
    WriteStringN(Str(MeasurementDelay))
    UseWindow(1)
    WriteStringN(Str(WindowX()))    
    WriteStringN(Str(WindowY()))
    UseWindow(0)
    WriteStringN(Str(WindowX()))
    WriteStringN(Str(WindowY()))
    
    WriteStringN(WBUStationID$)
    WriteStringN(WBUPassword$)
    WriteStringN(WBUServer$)
    WriteStringN(Str(WBUServerPort))
    WriteStringN(ProxyServer$)
    WriteStringN(Str(ProxyPort))
    WriteStringN(Str(UseProxy))
    WriteStringN(Str(EnableWBU))

    CloseFile(2)
  EndIf

Return

;-LoadPrefs

LoadPrefs:

  ; Set the default value is the pref file isn't found

  SelectedPort$     = "COM1"
  FrogAltitude      = 0
  OffsetPressure    = 0
  OffsetTemperature = 0
  OffsetHumidity    = 0
  MeasurementDelay  = 10

  WBUServer$    = "weather.weatherbyu.com"
  WBUServerPort = 80
  UseProxy      = 0
  
  If ReadFile(2, "PureFrog.prefs")

    SelectedPort$     = ReadString()
    FrogAltitude      = Val(ReadString())
    OffsetPressure    = Val(ReadString())
    OffsetTemperature = Val(ReadString())
    OffsetHumidity    = Val(ReadString())
    MeasurementDelay  = Val(ReadString())
    MonitorX          = Val(ReadString())
    MonitorY          = Val(ReadString())
    ConfigurationX    = Val(ReadString())
    ConfigurationY    = Val(ReadString())
    
    WBUStationID$     = ReadString()
    WBUPassword$      = ReadString()
    WBUServer$        = ReadString()
    WBUServerPort     = Val(ReadString())
    ProxyServer$      = ReadString()
    ProxyPort         = Val(ReadString())
    UseProxy          = Val(ReadString())
    EnableWBU         = Val(ReadString())
      
    CloseFile(2)
  EndIf

Return 
; ExecutableFormat=Windows
; EnableAsm
; EnableNT4
; UseIcon=PureFrog.ico
; Executable=E:\Documents and Settings\Administrateur\Bureau\PureFrog\PureFrog\PureFrog.exe
; EOF